<?php
/**
 * Lazy loading functions.
 *
 * @package Woodmart
 */

if ( ! defined( 'WOODMART_THEME_DIR' ) ) {
	exit( 'No direct script access allowed' );
}

if ( ! function_exists( 'woodmart_lazy_loading_init' ) ) {
	/**
	 * Init lazy loading.
	 *
	 * @param boolean $force_init Force init.
	 * @return void
	 */
	function woodmart_lazy_loading_init( $force_init = false ) {
		if ( ( ! woodmart_get_opt( 'lazy_loading' ) || is_admin() ) && ! $force_init ) {
			return;
		}

		// Used for avatar images.
		add_filter( 'get_avatar', 'woodmart_lazy_image_standard', 10 );

		// Used for instagram images.
		add_filter( 'woodmart_image', 'woodmart_lazy_image_standard', 10, 1 );

		// Images generated by WPBakery functions.
		add_filter( 'vc_wpb_getimagesize', 'woodmart_lazy_image', 10, 3 );

		// Products, blog, a lot of other standard WordPress images.
		add_filter( 'wp_get_attachment_image_attributes', 'woodmart_lazy_attributes', 10, 3 );

		// Elementor.
		add_filter( 'elementor/image_size/get_attachment_image_html', 'woodmart_filter_elementor_images', 10, 4 );
	}

	add_action( 'init', 'woodmart_lazy_loading_init', 120 );

}

if ( ! function_exists( 'woodmart_filter_elementor_images' ) ) {
	/**
	 * Filters HTML <img> tag and adds lazy loading attributes. Used for elementor images.
	 *
	 * @since 1.0.0
	 *
	 * @param string $html           Image html.
	 * @param array  $settings       Control settings.
	 * @param string $image_size_key Optional. Settings key for image size.
	 * @param string $image_key      Optional. Settings key for image.
	 *
	 * @return string
	 */
	function woodmart_filter_elementor_images( $html, $settings, $image_size_key, $image_key ) {
		if ( preg_match( "/src=['\"]data:image/is", $html ) ) {
			return $html;
		}

		$image         = $settings[ $image_key ];
		$image_sizes   = get_intermediate_image_sizes();
		$image_sizes[] = 'full';
		$size          = $settings[ $image_size_key . '_size' ];

		if ( $image['id'] && in_array( $size, $image_sizes ) ) { // phpcs:ignore
			return $html;
		}

		woodmart_enqueue_js_script( 'lazy-loading' );

		return woodmart_lazy_replace_image( $html, woodmart_lazy_get_default_preview() );
	}
}

if ( ! function_exists( 'woodmart_lazy_loading_rss_deinit' ) ) {
	/**
	 * Lazy loading deinit for RSS.
	 *
	 * @return void
	 */
	function woodmart_lazy_loading_rss_deinit() {
		if ( is_feed() ) {
			woodmart_lazy_loading_deinit( true );
		}
	}

	add_action( 'wp', 'woodmart_lazy_loading_rss_deinit', 10 );
}

if ( ! function_exists( 'woodmart_lazy_loading_deinit' ) ) {
	/**
	 * Lazy loading deinit.
	 *
	 * @param boolean $force_deinit Force deinit.
	 * @return void
	 */
	function woodmart_lazy_loading_deinit( $force_deinit = false ) {
		if ( woodmart_get_opt( 'lazy_loading' ) && ! $force_deinit ) {
			return;
		}

		remove_action( 'get_avatar', 'woodmart_lazy_image_standard', 10 );
		remove_action( 'woodmart_image', 'woodmart_lazy_image_standard', 10 );
		remove_action( 'vc_wpb_getimagesize', 'woodmart_lazy_image', 10 );
		remove_action( 'wp_get_attachment_image_attributes', 'woodmart_lazy_attributes', 10 );
		remove_action( 'elementor/image_size/get_attachment_image_html', 'woodmart_filter_elementor_images', 10 );
	}
}

if ( ! function_exists( 'woodmart_lazy_loading_force_deinit' ) ) {
	/**
	 * Fix Woocommerce email with lazy load.
	 *
	 * @return void
	 */
	function woodmart_lazy_loading_force_deinit() {
		woodmart_lazy_loading_deinit( true );
	}

	add_action( 'woocommerce_email_before_order_table', 'woodmart_lazy_loading_force_deinit', 20 );
	add_action( 'woocommerce_email_header', 'woodmart_lazy_loading_force_deinit', 20 );
	add_action( 'woocommerce_before_mini_cart_contents', 'woodmart_lazy_loading_force_deinit', 20 );
}

if ( ! function_exists( 'woodmart_lazy_loading_force_init' ) ) {
	/**
	 * Fix Woocommerce email with lazy load.
	 *
	 * @return void
	 */
	function woodmart_lazy_loading_force_init() {
		woodmart_lazy_loading_init( true );
	}

	add_action( 'woocommerce_email_after_order_table', 'woodmart_lazy_loading_force_init', 20 );
	add_action( 'woocommerce_email_footer', 'woodmart_lazy_loading_force_init', 20 );
	add_action( 'woocommerce_mini_cart_contents', 'woodmart_lazy_loading_init', 20 );
}

if ( ! function_exists( 'woodmart_lazy_image_standard' ) ) {
	/**
	 * Filters HTML <img> tag and adds lazy loading attributes. Used for instagram images.
	 *
	 * @param string $html Image html.
	 * @return string
	 */
	function woodmart_lazy_image_standard( $html ) {
		if ( preg_match( "/src=['\"]data:image/is", $html ) ) {
			return $html;
		}

		woodmart_enqueue_js_script( 'lazy-loading' );

		return woodmart_lazy_replace_image( $html, woodmart_lazy_get_default_preview() );
	}
}

if ( ! function_exists( 'woodmart_lazy_get_default_preview' ) ) {
	/**
	 * Get default preview image.
	 *
	 * @return mixed|null
	 */
	function woodmart_lazy_get_default_preview() {
		$placeholder = woodmart_get_opt( 'lazy_custom_placeholder' );

		if ( ! empty( $placeholder['url'] ) ) {
			return $placeholder['url'];
		}

		return WOODMART_IMAGES . '/lazy.svg';
	}
}

if ( ! function_exists( 'woodmart_lazy_image' ) ) {
	/**
	 * Filters WPBakery generated image. Needs an HTML, its ID, and params with image size.
	 *
	 * @param array   $img Image.
	 * @param integer $attach_id Attachment ID.
	 * @param array   $params Params.
	 * @return array
	 */
	function woodmart_lazy_image( $img, $attach_id, $params ) {
		if ( preg_match( "/src=['\"]data:image/is", $img['thumbnail'] ) ) {
			return $img;
		}

		if ( ! empty( $params['thumb_size'] ) ) {
			$image_sizes   = get_intermediate_image_sizes();
			$image_sizes[] = 'full';

			if ( in_array( $params['thumb_size'], $image_sizes ) ) { // phpcs:ignore
				return $img;
			}
		}

		$img['thumbnail'] = woodmart_lazy_replace_image( $img['thumbnail'], woodmart_lazy_get_default_preview() );

		woodmart_enqueue_js_script( 'lazy-loading' );

		return $img;
	}
}

if ( ! function_exists( 'woodmart_lazy_replace_image' ) ) {
	/**
	 * Filters <img> tag passed as an argument.
	 *
	 * @param string $html HTML.
	 * @param string $src Image src.
	 * @return string
	 */
	function woodmart_lazy_replace_image( $html, $src ) {
		$class = woodmart_lazy_css_class();

		$new = preg_replace( '/<img(.*?)src=/is', '<img$1src="' . $src . '" data-wood-src=', $html );
		$new = preg_replace( '/<img(.*?)srcset=/is', '<img$1srcset="" data-srcset=', $new );

		if ( preg_match( '/class=["\']/i', $new ) ) {
			$new = preg_replace( '/class=(["\'])(.*?)["\']/is', 'class=$1' . $class . ' $2$1', $new );
		} else {
			$new = preg_replace( '/<img/is', '<img class="' . $class . '"', $new );
		}

		return $new;
	}
}

if ( ! function_exists( 'woodmart_lazy_attributes' ) ) {
	/**
	 * Filters default WordPress images ATTRIBUTES array called by core API functions.
	 *
	 * @param array  $attr Attributes image.
	 * @param object $attachment Attachment.
	 * @param string $size Size.
	 * @return array
	 */
	function woodmart_lazy_attributes( $attr, $attachment, $size ) {
		if ( wp_is_serving_rest_request() || $attr['src'] == woodmart_lazy_get_default_preview() ) {
			return $attr;
		}

		$attr['data-wood-src'] = $attr['src'];

		if ( ! empty( $attr['srcset'] ) ) {
			$attr['data-srcset'] = $attr['srcset'];
		}

		if ( is_object( $attachment ) ) {
			$attr['src'] = woodmart_lazy_get_default_preview();
		}

		$attr['srcset'] = '';

		$attr['class'] = $attr['class'] . woodmart_lazy_css_class();

		woodmart_enqueue_js_script( 'lazy-loading' );

		return $attr;
	}
}

if ( ! function_exists( 'woodmart_lazy_css_class' ) ) {
	/**
	 * Get lazy loading image CSS class.
	 *
	 * @return string
	 */
	function woodmart_lazy_css_class() {
		$class       = '';
		$lazy_effect = woodmart_get_opt( 'lazy_effect', 'none' );

		if ( 'none' !== $lazy_effect ) {
			$class .= ' wd-lazy-' . $lazy_effect;
		}

		$class .= woodmart_get_old_classes( ' wd-lazy-load woodmart-lazy-load' );

		return $class;
	}
}

if ( ! function_exists( 'woodmart_disable_default_lazy_loading' ) ) {
	/**
	 * Disable default lazy loading.
	 *
	 * @return void
	 */
	function woodmart_disable_default_lazy_loading() {
		if ( woodmart_get_opt( 'disable_wordpress_lazy_loading' ) ) {
			add_filter( 'wp_lazy_loading_enabled', '__return_false' );
		}
	}

	add_action( 'init', 'woodmart_disable_default_lazy_loading', 120 );
}
